<?php

namespace App\Http\Controllers;

use App\Imports\ProdukImport;
use App\Models\Kategori;
use App\Models\Produk;
use App\Models\Satuan;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;

// use Illuminate\Support\Facades\Route;
use Intervention\Image\Laravel\Facades\Image;

class ProdukController extends Controller
{
    public function index()
    {
        $produk = Produk::pencarian()->latest()->paginate(10);
        $data = [
            'title' => 'Ramizah Store | Produk',
            'page' => 'Produk',
            'menuActive' => 'produk',
            'menuOpen' => 'produk',
        ];
        return view('admin.produk.produk', compact('data', 'produk'))->with('no', 1);
    }

    public function download_produk()
    {
        return Storage::download('public/format/format_import_data_produk.xlsx');
    }

    public function import_produk(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xls,xlsx'
        ]);

        $file = $request->file('file');
        $nama_file = $file->hashName();
        $path = $file->storeAs('public/excel/', $nama_file);
        Excel::import(new ProdukImport(), storage_path('app/public/excel/' . $nama_file));
        Storage::delete($path);

        return redirect()->route('produk.index')->with(['success' => 'Import Data Berhasil Disimpan!']);
    }

    public function tambah()
    {
        $data = [
            'title' => 'Ramizah Store | Tambah Produk',
            'page' => 'Tambah Produk',
            'menuActive' => 'produk',
            'menuOpen' => 'produk',
        ];
        return view('admin.produk.tambah', compact('data'));
    }

    public function view_kategori()
    {
        $kategori = Kategori::latest()->get();
        echo '<option value="">Pilih</option>';
        foreach ($kategori as $k) {
            echo '<option value="' . $k->id . '">' . $k->nm_kategori . '</option>';
        }
    }

    public function simpan_kategori(Request $request)
    {
        $request->validate(
            [
                'nm_kategori'  => 'required|unique:kategoris,nm_kategori',
                'ktr_kategori'   => 'required',
            ],
            [
                'required' => 'Form wajib diisi!',
                'unique' => 'Username sudah pernah diinputkan, silahkan input username lain!',
            ]
        );

        Kategori::create([
            'nm_kategori'         => $request->nm_kategori,
            'ktr_kategori'   => $request->ktr_kategori,
        ]);

        return response()->json();
    }

    public function view_satuan()
    {
        $satuan = Satuan::latest()->get();
        echo '<option value="">Pilih</option>';
        foreach ($satuan as $s) {
            echo '<option value="' . $s->id . '">' . $s->nm_satuan . '</option>';
        }
    }

    public function simpan_satuan(Request $request)
    {
        $request->validate(
            [
                'nm_satuan'  => 'required|unique:kategoris,nm_kategori',
                'ktr_satuan'   => 'required',
            ],
            [
                'required' => 'Form wajib diisi!',
                'unique' => 'Satuan sudah pernah diinputkan, silahkan input Satuan lain!',
            ]
        );

        Satuan::create([
            'nm_satuan'         => $request->nm_satuan,
            'ktr_satuan'   => $request->ktr_satuan,
        ]);

        return response()->json();
    }

    public function store(Request $request): RedirectResponse
    {
        $request->validate(
            [
                'kategori' => 'required',
                'satuan' => 'required',
                'kd_produk' => 'required|unique:produks,kd_produk',
                'nm_produk' => 'required',
                'berat' => 'required',
                'harga_beli' => 'required',
                'harga_jual' => 'required',
                'diskon' => 'required',
                'keterangan' => 'required',
                'gambar'             => 'image|mimes:jpeg,jpg,png|max:2048',
            ],
            [
                'required' => 'Form wajib diisi!',
                'unique' => 'Kode Produk sudah pernah diinputkan, silahkan input Kode  lain!',
            ]
        );

        if ($request->hasFile('gambar')) {
            $upload = $request->file('gambar');
            $image = Image::read($upload)->resize(600, 600);
            Storage::put(
                'public/produk/' . $upload->hashName(),
                $image->encodeByExtension($upload->getClientOriginalExtension(), quality: 85)
            );
            $isi = $upload->hashName();
        } else {
            $isi = '';
        }

        Produk::create([
            'kategori_id'    => $request->kategori,
            'satuan_id' => $request->satuan,
            'kd_produk' => $request->kd_produk,
            'nm_produk' => $request->nm_produk,
            'berat' => $request->berat,
            'harga_beli' => $request->harga_beli,
            'harga_jual' => $request->harga_jual,
            'diskon' => $request->diskon,
            'ktr_produk' => $request->keterangan,
            'gambar'        => $isi,
            'link_produk'   => strtolower($request->kd_produk) . '-' . Str::slug($request->nm_produk),
        ]);

        return redirect()->route('produk.index')->with(['success' => 'Data Berhasil Disimpan!']);
    }

    public function edit($id)
    {
        $p = Produk::where('link_produk', $id)->first();
        $kategori = Kategori::latest()->get();
        $satuan = Satuan::latest()->get();
        $data = [
            'title' => 'MiniMarket | Edit Produk',
            'page' => 'Edit Produk',
            'menuActive' => 'produk',
            'menuOpen' => 'produk',
        ];
        return view('admin.produk.edit', compact('data', 'p', 'kategori', 'satuan'));
    }

    public function update(Request $request)
    {
        $id = $request->id;

        $gbrlama = $request->gbrlama;

        if ($request->kdlama != $request->kd_produk) {
            $rule = 'required|unique:produks,kd_produk';
        } else {
            $rule = 'required';
        }

        $request->validate(
            [
                'kategori' => 'required',
                'satuan' => 'required',
                'kd_produk' => $rule,
                'nm_produk' => 'required',
                'berat' => 'required',
                'harga_beli' => 'required',
                'harga_jual' => 'required',
                'diskon' => 'required',
                'keterangan' => 'required',
                'gambar'             => 'image|mimes:jpeg,jpg,png|max:2048',
            ],
            [
                'required' => 'Form wajib diisi!',
                'unique' => 'Kode Produk sudah pernah diinputkan, silahkan input Kode  lain!',
            ]
        );

        $produk = Produk::findOrFail($id);
        if ($request->hasFile('gambar')) {
            // //upload image
            if ($gbrlama != null) {
                Storage::delete('public/produk/' .  $gbrlama);
            }
            $upload = $request->file('gambar');
            $image = Image::read($upload)->resize(600, 600);
            Storage::put(
                'public/produk/' . $upload->hashName(),
                $image->encodeByExtension($upload->getClientOriginalExtension(), quality: 85)
            );
            $produk->update([
                'gambar'             =>  $upload->hashName(),
            ]);
        }

        $produk->update([
            'kategori_id'    => $request->kategori,
            'satuan_id' => $request->satuan,
            'kd_produk' => $request->kd_produk,
            'nm_produk' => $request->nm_produk,
            'berat' => $request->berat,
            'harga_beli' => $request->harga_beli,
            'harga_jual' => $request->harga_jual,
            'diskon' => $request->diskon,
            'ktr_produk' => $request->keterangan,
            'link_produk' => strtolower($request->kd_produk) . '-' . Str::slug($request->nm_produk),
        ]);

        return redirect()->route('produk.index')->with(['success' => 'Data Berhasil Disimpan!']);
    }

    public function diskon_produk(Request $request)
    {
        $id = $request->id;
        $produk = Produk::findOrFail($id);
        $produk->update([
            'diskon' => $request->diskon,
        ]);
        return redirect()->route('produk.index')->with(['success' => 'Edit Diskon Berhasil Disimpan!']);
    }

    public function gambar(Request $request)
    {
        $p = Produk::where('id', $request->id)->first();
        $data = [
            'nama' => $p->nm_produk,
            'gbrlama' => $p->gambar,
            'gambar' => '<img src="' . asset('/storage/produk/' . $p->gambar) . '" id="vall" width="100%" height="190px" alt="">',
        ];
        return response()->json($data);
    }

    public function upload_gambar(Request $request)
    {
        $gbrlama = $request->gbrlama;
        $produk = Produk::findOrFail($request->id);
        if ($request->hasFile('gambar')) {
            // //upload image
            if ($gbrlama != null) {
                Storage::delete('public/produk/' .  $gbrlama);
            }
            $upload = $request->file('gambar');
            $image = Image::read($upload)->resize(600, 600);
            Storage::put(
                'public/produk/' . $upload->hashName(),
                $image->encodeByExtension($upload->getClientOriginalExtension(), quality: 85)
            );
            $produk->update([
                'gambar'             =>  $upload->hashName(),
            ]);
        }
        return redirect()->route('produk.index')->with(['success' => 'Upload gambar berhasil!']);
    }

    public function detail(Request $request)
    {
        $p = Produk::where('id', $request->id)->first();
        return view('admin.produk.detail', compact('p'));
    }

    public function destroy_all_produk(Request $request)
    {
        $ids = $request->allVals;
        foreach ($ids as  $id) {
            $img = Produk::where('id', $id)->first();
            if ($img->gambar != null) {
                Storage::delete('public/produk/' .  $img->gambar);
            }
            Produk::where('id', $id)->delete();
        }

        return response()->json($ids);
    }

    public function cetak_barcode(Request $request)
    {
        $ids = $request->allVals;
        $barcode = [];
        foreach ($ids as $id) {
            $p = Produk::where('id', $id)->first();
            $barcode[] = [
                'nama' =>  $p->nm_produk,
                'kode' =>   $p->kd_produk,
                'harga' => 'Rp. ' . number_format($p->harga_jual, 0, ',', '.'),
            ];
        }

        return view('admin.produk.barcode', compact('barcode'))->with('no', '1');
    }

    public function cetak_induk(Request $request)
    {
        $kategori = Kategori::latest('id', 'DESC')->get();
        $satuan = Satuan::latest('id', 'DESC')->get();
        return view('admin.produk.induk', compact('kategori', 'satuan'));
    }
}
